# from django.shortcuts import render
import re
from apps.accounts.services.password_service import hash_password, bcrypt_password_hash

# Create your views here.
from rest_framework.decorators import api_view
from rest_framework.response import Response
from rest_framework import status

from apps.accounts.models.user import find_user_by_identifier, find_user_by_id, mark_user_verified, find_user_by_any_identifier, can_attempt_login, clear_login_attempts, register_failed_login
from apps.accounts.services.password_service import verify_password
from apps.auth.utils.jwt_service import generate_access_token, generate_refresh_token, verify_access_token, verify_refresh_token
from .serializer import serialize_user
from core.utils.validation import validate_name,validate_phone, validate_email_field, validate_identifier, validate_dob, validate_gender, validate_otp, validate_password
from django.core.exceptions import ValidationError

@api_view(["GET"])
def test_api(request):
    return Response(
        {"status": "Auth API working ✅"},
        status=status.HTTP_200_OK
    )




from apps.accounts.models.user import (
    create_user,
    find_user_by_email,
    find_user_by_phone
)

EMAIL_REGEX = r"^[\w\.-]+@[\w\.-]+\.\w+$"
PHONE_REGEX = r"^[6-9]\d{9}$"  # India-friendly

@api_view(["POST"])
def register_user(request):
    data = request.data

    # Mandatory fields
    required_fields = ["first_name", "phone_number", "primary_email", "password"]
    for field in required_fields:
        if not data.get(field):
            return Response(
                {"error": f"{field} is required"},
                status=status.HTTP_400_BAD_REQUEST
            )

    # Email validation
    # if not re.match(EMAIL_REGEX, data["primary_email"]):
    #     return Response(
    #         {"error": "Invalid email format"},
    #         status=status.HTTP_400_BAD_REQUEST
    #     )

    # # Phone validation
    # if not re.match(PHONE_REGEX, data["phone_number"]):
    #     return Response(
    #         {"error": "Invalid phone number"},
    #         status=status.HTTP_400_BAD_REQUEST
    #     )
    
    
    
    # validations
    
    try:
        validate_name(data["first_name"])
        validate_email_field(data["primary_email"])
        validate_phone(data["phone_number"])
        validate_password(data["password"])
    except ValidationError as e:
        return Response(
            {"error": e.message},
            status=status.HTTP_400_BAD_REQUEST
        )
    
    if data["last_name"]:
        try:
            validate_name(data["last_name"])
        except ValidationError as e:
             return Response(
            {"error": "Invalid Last Name"},
            status=status.HTTP_400_BAD_REQUEST
        )
             
    # Duplicate checks
    if find_user_by_email(data["primary_email"]):
        return Response(
            {"error": "Email already registered"},
            status=status.HTTP_409_CONFLICT
        )

    if find_user_by_phone(data["phone_number"]):
        return Response(
            {"error": "Phone number already registered"},
            status=status.HTTP_409_CONFLICT
        )

    hashed_password = hash_password(data["password"])
    hashed_chitraplay_password = bcrypt_password_hash(data["password"])
    user_payload = {
        "first_name": data["first_name"].strip(),
        "last_name": data.get("last_name", "").strip(),
        "phone_number": data["phone_number"],
        "primary_email": data["primary_email"].lower(),
        "recovery_phone": data.get("recovery_phone"),
        "recovery_email": data.get("recovery_email"),
        "password": hashed_password,
        "chitraplay_password" : hashed_chitraplay_password,
        "address": data.get("address"),
    }


    user_id = create_user(user_payload)

    return Response(
        {
            "message": "User registered successfully",
            "user_id": user_id
        },
        status=status.HTTP_201_CREATED
    )



from core.db.mongo import user_sessions_collection
from bson import ObjectId


from core.db.mongo import user_sessions_collection
from .utils.device import parse_device
from datetime import datetime, timezone
from hashlib import sha256

@api_view(["POST"])
def login_user(request):
    data = request.data

    identifier = data.get("identifier")  # email or phone
    password = data.get("password")
    device_id = request.headers.get("X-Device-Id")
    user_agent = request.META.get("HTTP_USER_AGENT", "")
    ip = request.META.get("REMOTE_ADDR")
    
    if not device_id:
        return Response(
        {"error": "Device ID missing"},
        status=400
    )
        
    if not identifier or not password:
        return Response(
            {"error": "Identifier and password required"},
            status=status.HTTP_400_BAD_REQUEST
        )

    try:
        validate_identifier(identifier)
    except ValidationError as e:
        return Response(
            {"error":e.message},
            status=status.HTTP_400_BAD_REQUEST
        )
    try:
        validate_password(password)
    except ValidationError as e:
        return Response(
            {"error": "Invalid credentials"},
            status=status.HTTP_401_UNAUTHORIZED
        )
        
    user = find_user_by_identifier(identifier)

    if not user:
        return Response(
            {"error": "Invalid credentials"},
            status=status.HTTP_401_UNAUTHORIZED
        )

    user_id = str(user["_id"])
    # 🔒 Brute force check BEFORE password verify
    allowed, error = can_attempt_login(user_id, ip)
    if not allowed:
        return Response({"error": error}, status=403)
    
    if not user.get("is_verified", False):
        return Response(
        {"error": "Account not verified. Please verify your account before logging in."},
        status=status.HTTP_403_FORBIDDEN
    )
        
        
    if not user.get("is_active", False):
        return Response(
        {"error": "Account has been suspended"},
        status=status.HTTP_403_FORBIDDEN
    )
        
    if not verify_password(user["password"], password):
        register_failed_login(str(user["_id"]), ip)
        return Response(
            {"error": "Invalid credentials"},
            status=status.HTTP_401_UNAUTHORIZED
        )
    


    user_id = str(user["_id"])
    session_id = str(uuid4()) 
    access_token = generate_access_token(user_id,session_id )
    refresh_token = generate_refresh_token(user_id, session_id)
    
    
    # =======================
    # 🔐 DEVICE SESSION LOGIC
    # =======================



    device_info = parse_device(user_agent)

    refresh_token_hash = sha256(refresh_token.encode()).hexdigest()

    
    
    user_sessions_collection.insert_one({
        "user_id": ObjectId(user_id),
        "session_id": session_id,
        "device_id": device_id,
        "ip_address": ip,
        "browser": device_info["browser"],
        "os": device_info["os"],
        "user_agent": user_agent,
        "refresh_token_hash": refresh_token_hash,
        "created_at": datetime.now(timezone.utc),
        "last_active": datetime.now(timezone.utc),
        "is_active": True
    })
    clear_login_attempts(user_id, ip)

    response = Response(
        {
            "message": "Login successful",
            "user": {
                "id": user_id,
                "first_name": user["first_name"],
                "primary_email": user["primary_email"],
                "phone_number": user["phone_number"]
            }
        },
        status=status.HTTP_200_OK
    )

    # HttpOnly secure cookies
    response.set_cookie(
        key="access_token",
        value=access_token,
        httponly=True,
        secure=False,  # True in production
        samesite="Lax",
        max_age=900
    )

    response.set_cookie(
        key="refresh_token",
        value=refresh_token,
        httponly=True,
        secure=False,
        samesite="Lax",
        max_age=604800
    )

    return response





from core.db.mongo import user_sessions_collection

@api_view(["GET"])
def me(request):
    token = request.COOKIES.get("access_token")

    # Optional: allow Authorization header
    if not token:
        auth_header = request.headers.get("Authorization")
        if auth_header and auth_header.startswith("Bearer "):
            token = auth_header.split(" ")[1]

    if not token:
        return Response(
            {"authenticated": False},
            status=status.HTTP_401_UNAUTHORIZED
        )

    payload = verify_access_token(token)
    print("JWT PAYLOAD:", payload)


    if not payload:
        return Response(
            {"authenticated": False},
            status=status.HTTP_401_UNAUTHORIZED
        )

    user_id = payload.get("sub")
    session_id = payload.get("session_id")
    print(session_id)
    
    if not user_id or not session_id:
        return Response(
            {"authenticated": False},
            status=status.HTTP_401_UNAUTHORIZED
        )

    # 🔐 SESSION VALIDATION
    session = user_sessions_collection.find_one({
        "user_id": ObjectId(user_id),
        "session_id": session_id,
        "is_active": True
    })

    if not session:
        return Response(
            {
                "authenticated": False,
                "reason": "Session revoked"
            },
            status=status.HTTP_401_UNAUTHORIZED
        )


    if not user_id:
        return Response(
            {"authenticated": False},
            status=status.HTTP_401_UNAUTHORIZED
        )
        
    user = find_user_by_id(user_id)

    if not user or not user.get("is_active", False):
        return Response(
            {"authenticated": False},
            status=status.HTTP_401_UNAUTHORIZED
        )

    return Response({
        "authenticated": True,
        "user":  serialize_user(user)
    })




from .decorators import jwt_required
from .utils.session import deactivate_session, hash_refresh_token, deactivate_all_sessions, deactivate_specific_device


@api_view(["POST"])
@jwt_required
def logout_current_device(request):

    refresh_token = request.COOKIES.get("refresh_token")

    if not refresh_token:
        return Response({"error": "Refresh token missing"}, status=400)

    refresh_token_hash = hash_refresh_token(refresh_token)

    deactivate_session(request.user_id, refresh_token_hash)

    response = Response({"message": "Logged out from current device"})
    response.delete_cookie("access_token")
    response.delete_cookie("refresh_token")

    return response


@api_view(["POST"])
@jwt_required
def logout_all_devices(request):
    deactivate_all_sessions(request.user_id)

    response = Response({"message": "Logged out from all devices"})
    response.delete_cookie("access_token")
    response.delete_cookie("refresh_token")

    return response


@api_view(["POST"])
@jwt_required
def logout_specific_device(request):
    device_id = request.data.get("device_id")
    session_id = request.data.get("session_id")

    if not device_id and not session_id:
        return Response({"error": "device id and session id required"}, status=400)

    result = deactivate_specific_device(request.user_id, device_id, session_id)

    if result.matched_count == 0:
        return Response({"error": "Device not found or already logged out"}, status=404)

    return Response({"message": "Device logged out successfully"})





from uuid import uuid4
from hashlib import sha256

@api_view(["POST"])
def refresh_token(request):
    refresh_token = request.COOKIES.get("refresh_token")

    if not refresh_token:
        return Response({"error": "Unauthorized"}, status=401)

    payload = verify_refresh_token(refresh_token)

    if not payload:
        return Response({"error": "Invalid refresh token"}, status=401)

    refresh_hash = sha256(refresh_token.encode()).hexdigest()

    session = user_sessions_collection.find_one({
        "session_id": payload["session_id"],
        "refresh_token_hash": refresh_hash,
        "is_active": True
    })

    if not session:
        return Response({"error": "Session revoked"}, status=401)

    # 🔁 ROTATE
    new_session_id = str(uuid4())
    new_refresh = generate_refresh_token(payload["user_id"], new_session_id)
    new_access = generate_access_token(payload["user_id"], new_session_id)

    user_sessions_collection.update_one(
        {"_id": session["_id"]},
        {"$set": {
            "session_id": new_session_id,
            "refresh_token_hash": sha256(new_refresh.encode()).hexdigest(),
            "last_active": datetime.now(timezone.utc)
        }}
    )

    response = Response({"message": "Token refreshed"})
    response.set_cookie("access_token", new_access, httponly=True)
    response.set_cookie("refresh_token", new_refresh, httponly=True)

    return response





from .models.otp import can_send_otp, save_otp, is_ip_blocked
from .utils.otp import generate_otp, hash_otp



@api_view(["POST"])
def send_otp(request):
    ip = request.META.get("REMOTE_ADDR")
    identifier = request.data.get("identifier")

    if not identifier:
        return Response(
            {"error": "Email or phone is required"},
            status=status.HTTP_400_BAD_REQUEST
        )

    try:
        validate_identifier(identifier)
    except ValidationError as e:
        return Response(
            {"error": e.message},
            status=status.HTTP_400_BAD_REQUEST
        )
        
    # 1️⃣ Find user
    user = find_user_by_identifier(identifier)
    if not user:
        return Response(
            {"error": "User not found"},
            status=status.HTTP_404_NOT_FOUND
        )

    if user.get("is_verified"):
        return Response(
            {"error": "User already verified"},
            status=status.HTTP_400_BAD_REQUEST
        )

    # 2️⃣ OTP rate limit + blocking
    allowed, error = can_send_otp(str(user["_id"]), ip, user["phone_number"])
    if not allowed:
        return Response(
            {"error": error},
            status=status.HTTP_429_TOO_MANY_REQUESTS
        )

    # 3️⃣ Generate OTP
    otp = generate_otp()
    otp_hash = hash_otp(otp)

    save_otp(str(user["_id"]), otp_hash, ip,to_verify=identifier, identifier=identifier)

    # REMOVE THIS IN PRODUCTION
    # print("OTP:", otp)
    send_otp_sms(user["phone_number"], otp)

    return Response(
        {"message": "OTP sent successfully"},
        status=status.HTTP_200_OK
    )



from .models.otp import (
    get_valid_otp,
    increment_verify_attempt,
    mark_otp_verified,
    delete_all_user_otps,
    clear_ip_block,
    MAX_VERIFY_ATTEMPTS
)

from .utils.otp import verify_otp_hash


@api_view(["POST"])
def verify_otp(request):
    ip = request.META.get("REMOTE_ADDR")
    identifier = request.data.get("identifier")
    otp_input = request.data.get("otp")

    if not identifier or not otp_input:
        return Response(
            {"error": "Identifier and OTP are required"},
            status=status.HTTP_400_BAD_REQUEST
        )

    try:
        validate_identifier(identifier)
        validate_otp(otp_input)
    except ValidationError as e:
        return Response(
            {"error": e.message},
            status=status.HTTP_400_BAD_REQUEST
        )
    
    # try:
    #     validate_otp(otp_input)
    # except ValidationError as e:
    #     return Response(
    #         {"error": e.message},
    #         status=status.HTTP_400_BAD_REQUEST
    #     )
        
    user = find_user_by_identifier(identifier)
    if not user:
        return Response(
            {"error": "Invalid OTP"},
            status=status.HTTP_400_BAD_REQUEST
        )

    if user.get("is_verified"):
        return Response(
            {"error": "User already verified"},
            status=status.HTTP_400_BAD_REQUEST
        )

    otp_record = get_valid_otp(str(user["_id"]), identifier, identifier)
    if not otp_record:
        return Response(
            {"error": "OTP expired or invalid"},
            status=status.HTTP_400_BAD_REQUEST
        )

    # Check attempts
    attempts = otp_record.get("verify_attempts", 0)
    if attempts >= MAX_VERIFY_ATTEMPTS:
        return Response(
            {"error": "Too many wrong attempts"},
            status=status.HTTP_429_TOO_MANY_REQUESTS
        )

    # Verify OTP
    if not verify_otp_hash(otp_input, otp_record["otp_hash"]):
        increment_verify_attempt(otp_record["_id"])
        return Response(
            {"error": "Invalid OTP"},
            status=status.HTTP_400_BAD_REQUEST
        )

    # ✅ SUCCESS FLOW
    mark_otp_verified(otp_record["_id"])
    delete_all_user_otps(str(user["_id"]))
    clear_ip_block(ip, str(user["_id"]))
    mark_user_verified(str(user["_id"]))

    return Response(
        {"message": "OTP verified successfully"},
        status=status.HTTP_200_OK
    )



class OtpPurpose:
    LOGIN = "LOGIN"
    EMAIL_VERIFICATION = "EMAIL_VERIFICATION"
    PHONE_VERIFICATION = "PHONE_VERIFICATION"
    RECOVERY_EMAIL = "RECOVERY_EMAIL"
    PASSWORD_RESET = "PASSWORD_RESET"
    ACCOUNT_RECOVERY = "ACCOUNT_RECOVERY"




from core.db.mongo import users_collection, otp_collection


# @api_view(["POST"])
# def send_otp_common(request, purpose):
#     ip = request.META.get("REMOTE_ADDR")
#     identifier = request.data.get("identifier")

#     if not identifier:
#         return Response({"error": "Identifier required"}, status=400)

#     user = find_user_by_any_identifier(identifier)
#     if not user:
#         return Response({"error": "User not found"}, status=404)

#     allowed, error = can_send_otp(str(user["_id"]), ip, identifier=identifier, purpose=purpose)
#     if not allowed:
#         return Response(
#             {"error": error},
#             status=status.HTTP_429_TOO_MANY_REQUESTS
#         )

#     otp = generate_otp()
#     otp_hash = hash_otp(otp)

#     save_otp(
#         user_id=str(user["_id"]),
#         otp_hash=otp_hash,
#         ip=ip,
#         purpose=purpose,
#         identifier=identifier
#     )

#     # print("OTP:", otp)  # REMOVE IN PROD

#     return Response({"message": "OTP sent"}, status=status.HTTP_200_OK)


from .utils.validators import is_valid_email, is_valid_phone 


def purpose_validation_check(user, purpose, identifier):
    
    # 🔒 PURPOSE BASED VALIDATION
    purpose = purpose.upper()    
    if purpose == "PRIMARY_EMAIL":
        if not is_valid_email(identifier):
            return {"is_valid" :False, "error": "Invalid email format", }
        
    if purpose == "RECOVERY_EMAIL":
        if not is_valid_email(identifier):
            return {"is_valid" :False, "error": "Invalid email format", }
            
        # if identifier != user.get("recovery_email"):
        #     return Response(
        #         {"error": "Email does not match recovery email"},
        #         status=403
        #     )

    elif purpose == "RECOVERY_PHONE":
        if not is_valid_phone(identifier):
            return {"is_valid" :False, "error": "Invalid phone number", }

        # if identifier != user.get("recovery_phone"):
        #     return Response(
        #         {"error": "Phone number does not match recovery phone"},
        #         status=403
        #     )

    # else:
    #     return Response(
    #         {"error": "Invalid OTP purpose"},
    #         status=400
    #     )
    return {"is_valid" : True}
        
        
from .utils.email_service import send_otp_email
from .utils.sms_service import send_otp_sms
from config.settings.base import DEFAULT_FROM_EMAIL

def send_recovery_otp(request, purpose):
    ip = request.META.get("REMOTE_ADDR")
    identifier = request.data.get("identifier")
    user_id = request.user_id

    if not identifier:
        return Response({"error": "Identifier required"}, status=400)

    try:
        validate_identifier(identifier)
    except ValidationError as e:
        return Response(
            {"error": e.message},
            status=status.HTTP_400_BAD_REQUEST
        )

        
    user = find_user_by_id(user_id)

    if not user:
        return Response({"error": "User not found"}, status=404)
    
    if not user or not user.get("is_active", False):
        return Response(
            {"authenticated": False},
            status=status.HTTP_401_UNAUTHORIZED
        )
        
    is_valid = purpose_validation_check(user, purpose, identifier)
    
    if not is_valid["is_valid"]:
         return Response(
            {"error": is_valid["error"]},
            status=400
        )
        
    allowed, error = can_send_otp(str(user["_id"]), ip, identifier=identifier, purpose=purpose)
    if not allowed:
        return Response(
            {"error": error},
            status=status.HTTP_429_TOO_MANY_REQUESTS
        )

    otp = generate_otp()
    otp_hash = hash_otp(otp)


    purpose = purpose.lower()

    # Existing value in DB
    old_value = user.get(purpose)

    # 🔀 Decide OTP destination
    send_to = None
    send_via = None


    # ================= EMAIL FLOW =================
    if purpose in ["primary_email", "recovery_email"]:
        if old_value is None or old_value != identifier:
            # New / changed email → OTP to PHONE
            send_to = user.get("phone_number")
            send_via = "phone"
            send_otp_sms(send_to, otp)
        else:
            # Same email → OTP to EMAIL
            send_to = identifier
            send_via = "email"
            send_otp_email(
                to_email=send_to,
                otp=otp,
                sender_email= DEFAULT_FROM_EMAIL,
                sender_name="INDZS Security",
                purpose="Email Verification"
            )

    # ================= PHONE FLOW =================
    elif purpose == "recovery_phone":
        if old_value is None or old_value != identifier:
            # New / changed phone → OTP to EMAIL
            send_to = user.get("primary_email")
            send_via = "email"
            send_otp_email(
                to_email=send_to,
                otp=otp,
                sender_email= DEFAULT_FROM_EMAIL,
                sender_name="INDZS Security",
                purpose="Phone Verification"
            )
        else:
            # Same phone → OTP to PHONE
            send_to = identifier
            send_via = "phone"
            send_otp_sms(send_to, otp)
            
            
    save_otp(
        user_id=str(user["_id"]),
        otp_hash=otp_hash,
        ip=ip,
        to_verify=identifier,
        identifier=send_to,
        purpose=purpose,
    )

    # print("OTP:", otp)  # REMOVE IN PROD

    return Response({"message": "OTP sent", "send_to" : send_to}, status=status.HTTP_200_OK)




@api_view(["POST"])
@jwt_required
def send_recovery_email_otp(request):
    purpose = request.data.get("purpose")
    if not purpose:
        return Response({"error": "Purpose required"}, status=400)
    return send_recovery_otp(request, purpose)

    


# @api_view(["POST"])
# def send_recovery_phone_otp(request):
#     return send_otp_common(request, "recovery_phone")




# def update_user_verification(user_id, identifier, purpose, updated_verified = False):
#     update = {}
#     update[purpose] = identifier
    
#     if purpose == "primary_email":
#         update["primary_email_verified"] = updated_verified

#     elif purpose == "recovery_email":
#         update["recovery_email_verified"] = updated_verified

#     elif purpose == "recovery_phone":
#         update["recovery_phone_verified"] = updated_verified

#     users_collection.update_one(
#         {"_id": ObjectId(user_id)},
#         {"$set": update}
#     )




def update_user_verification(user_id, identifier, purpose, updated_verified=False):
    """
    Rules:
    1. If field value changes -> verification = False
    2. If field is same and only verification update -> verification = updated_verified
    3. If old value is None and new value is added -> verification = False
    """

    user = users_collection.find_one({"_id": ObjectId(user_id)})
    if not user:
        return False

    update = {}

    # Map verification field
    verification_field_map = {
        "primary_email": "primary_email_verified",
        "recovery_email": "recovery_email_verified",
        "recovery_phone": "recovery_phone_verified",
    }

    verification_field = verification_field_map.get(purpose)

    old_value = user.get(purpose)

    # 🔍 Case 1: New value added (previously null)
    if old_value is None and identifier is not None:
        update[purpose] = identifier
        if verification_field:
            update[verification_field] = False

    # 🔄 Case 2: Value changed
    elif old_value != identifier:
        update[purpose] = identifier
        if verification_field:
            update[verification_field] = False

    # ✅ Case 3: Value same → only verification update allowed
    else:
        if verification_field:
            update[verification_field] = updated_verified

    if update:
        users_collection.update_one(
            {"_id": ObjectId(user_id)},
            {"$set": update}
        )

    return True




    
@api_view(["POST"])
@jwt_required
def verify_recovery_otp(request):
    ip = request.META.get("REMOTE_ADDR")
    identifier = request.data.get("identifier")
    otp_input = request.data.get("otp")
    purpose = request.data.get("purpose")
    user_id = request.user_id
    to_verify = request.data.get("to_verify")
    
    print(request.data)
    # print(user_id)
    
    
    if not all([identifier, otp_input, purpose, to_verify]):
        return Response({"error": "Missing fields"}, status=400)
    
    try:
        validate_identifier(identifier)
        validate_otp(otp_input)
        validate_identifier(to_verify)
    except ValidationError as e:
        return Response(
            {"error": e.message},
            status=status.HTTP_400_BAD_REQUEST
        )
        
    # try:
    #     validate_otp(otp_input)
    # except ValidationError as e:
    #     return Response(
    #         {"error": e.message},
    #         status=status.HTTP_400_BAD_REQUEST
    #     )
        
    # try:
    #     validate_identifier(to_verify)
    # except ValidationError as e:
    #      return Response(
    #         {"error": e.message},
    #         status=status.HTTP_400_BAD_REQUEST
    #     )
        
    user = find_user_by_id(user_id)


    if not user:
        return Response({"error": "Invalid OTP"}, status=400)
    
    if not user or not user.get("is_active", False):
        return Response(
            {"authenticated": False},
            status=status.HTTP_401_UNAUTHORIZED
        )

    otp_record = get_valid_otp(str(user["_id"]),to_verify, identifier , purpose)
    print(otp_record)
    
    if not otp_record:
        return Response({"error": "OTP expired"}, status=400)

    attempts = otp_record.get("verify_attempts", 0)
    if attempts >= MAX_VERIFY_ATTEMPTS:
        return Response({"error": "Too many attempts"}, status=429)

    if not verify_otp_hash(otp_input, otp_record["otp_hash"]):
        increment_verify_attempt(otp_record["_id"])
        return Response({"error": "Invalid OTP"}, status=400)

    # ✅ SUCCESS
    mark_otp_verified(otp_record["_id"])
    delete_all_user_otps(str(user["_id"]), purpose)
    clear_ip_block(ip, str(user["_id"]))

    update_user_verification(user["_id"], to_verify, purpose, True)

    return Response({"message": "OTP verified"}, status=status.HTTP_200_OK)







@api_view(["POST"])
def send_forgot_password_otp(request):
    ip = request.META.get("REMOTE_ADDR")
    identifier = request.data.get("identifier")  # email or phone

    if not identifier:
        return Response({"error": "Email or phone required"}, status=400)

    try:
        validate_identifier(identifier)
    except ValidationError as e:
         return Response(
            {"error": e.message},
            status=status.HTTP_400_BAD_REQUEST
        )
         
    user = find_user_by_any_identifier(identifier)
    if not user or not user.get("is_active", False):
        return Response({"error": "User not found"}, status=404)

    allowed, error = can_send_otp(str(user["_id"]), ip,identifier=identifier, purpose= OtpPurpose.PASSWORD_RESET)
    if not allowed:
        return Response({"error": error}, status=429)

    otp = generate_otp()
    otp_hash = hash_otp(otp)

    save_otp(
        user_id=str(user["_id"]),
        otp_hash=otp_hash,
        ip=ip,
        purpose=OtpPurpose.PASSWORD_RESET,
        identifier=identifier,
        to_verify=identifier,
    )

    print("FORGOT PASSWORD OTP:", otp)  # REMOVE IN PROD

    return Response({"message": "OTP sent for password reset"}, status=200)



@api_view(["POST"])
def verify_forgot_password_otp(request):
    identifier = request.data.get("identifier")
    otp_input = request.data.get("otp")

    if not identifier or not otp_input:
        return Response({"error": "Identifier and OTP required"}, status=400)

    try:
        validate_identifier(identifier)
        validate_otp(otp_input)
    except ValidationError as e:
         return Response(
            {"error": e.message},
            status=status.HTTP_400_BAD_REQUEST
        )
    
    # try:
    #     validate_otp(otp_input)
    # except ValidationError as e:
    #      return Response(
    #         {"error": e.message},
    #         status=status.HTTP_400_BAD_REQUEST
    #     )
         
    user = find_user_by_any_identifier(identifier)
    # print(user)
    if not user:
        return Response({"error": "Invalid OTP"}, status=400)

    otp_record = get_valid_otp(
        user_id=str(user["_id"]),
        to_verify=identifier,
        identifier=identifier,
        purpose=OtpPurpose.PASSWORD_RESET
    )

    if not otp_record:
        return Response({"error": "OTP expired or invalid"}, status=400)

    if otp_record.get("verify_attempts", 0) >= MAX_VERIFY_ATTEMPTS:
        return Response({"error": "Too many attempts"}, status=429)

    print(otp_record["otp_hash"])
    if not verify_otp_hash(otp_input, otp_record["otp_hash"]):
        increment_verify_attempt(otp_record["_id"])
        return Response({"error": "Invalid OTP"}, status=400)

    # ✅ Mark verified ONLY
    mark_otp_verified(otp_record["_id"])

    return Response({
        "message": "OTP verified. You may now reset password"
    }, status=200)



from apps.accounts.views import update_password_all_apps

@api_view(["POST"])
def reset_password(request):
    identifier = request.data.get("identifier")
    new_password = request.data.get("new_password")

    
    if not identifier or not new_password:
        return Response({"error": "Missing fields"}, status=400)

    try:
        validate_identifier(identifier)
        validate_password(new_password)
    except ValidationError as e:
         return Response(
            {"error": e.message},
            status=status.HTTP_400_BAD_REQUEST
        )
    
    user = find_user_by_any_identifier(identifier)
    if not user:
        return Response({"error": "Invalid request"}, status=400)

    otp_record = otp_collection.find_one({
        "user_id": user["_id"],
        "purpose": OtpPurpose.PASSWORD_RESET,
        "verified": True,
        "expires_at": {"$gte": datetime.now(timezone.utc)}
    })

    if not otp_record:
        return Response(
            {"error": "OTP verification required"},
            status=403
        )

    # 🔐 Update password
    hashed_password = hash_password(new_password)
    hashed_chitraplay_password = bcrypt_password_hash(new_password)
    users_collection.update_one(
        {"_id": user["_id"]},
        {"$set": {
                "password": hashed_password,
                "chitraplay_password" : hashed_chitraplay_password,
                }
         }
    )
    update_password_all_apps(user,new_password)

    # 🧹 Cleanup OTP
    otp_collection.delete_many({
        "user_id": user["_id"],
        "purpose": OtpPurpose.PASSWORD_RESET
    })

    # 🔥 Kill all sessions (IMPORTANT)
    user_sessions_collection.update_many(
        {"user_id": user["_id"]},
        {"$set": {"is_active": False}}
    )

    return Response({"message": "Password updated successfully"}, status=200)

def mask_email(email):
    if not email or "@" not in email:
        return None
    name, domain = email.split("@")
    return name[:2] + "*" * max(len(name) - 2, 1) + "@" + domain


def mask_phone(phone):
    if not phone or len(phone) < 4:
        return None
    return "*" * (len(phone) - 4) + phone[-4:]


@api_view(["POST"])
def try_another_way(request):
    identifier = request.data.get("identifier")

    if not identifier:
        return Response(
            {"error": "identifier required"},
            status=400
        )

    # 🔍 Find user by any known identifier
    user = users_collection.find_one({
        "$or": [
            {"primary_email": identifier},
            {"phone_number": identifier},
            {"recovery_email": identifier},
            {"recovery_phone": identifier},
        ]
    })

    # 🔐 Always return safely
    if not user:
        return Response({"methods": []}, status=200)

    methods = []

    # 📱 Phone number
    if user.get("phone_number"):
        methods.append({
            "type": "PHONE",
            "label": "SMS",
            "value": mask_phone(user["phone_number"])
        })
        
    # 📧 Primary email
    if user.get("primary_email"):
        methods.append({
            "type": "PRIMARY_EMAIL",
            "label": "Email",
            "value": mask_email(user["primary_email"])
        })

    # 📱 Recovery phone
    if user.get("recovery_phone"):
        methods.append({
            "type": "RECOVERY_PHONE",
            "label": "Recovery Phone",
            "value": mask_phone(user["recovery_phone"])
        })
        
    # 📧 Recovery email
    if user.get("recovery_email"):
        methods.append({
            "type": "RECOVERY_EMAIL",
            "label": "Recovery Email",
            "value": mask_email(user["recovery_email"])
        })



    return Response(
        {
            "methods": methods
        },
        status=200
    )



import uuid
from datetime import timedelta, timezone

from core.db.mongo import account_recovery_otp_collection
from .models.otp import can_send_account_otp,save_account_otp
from datetime import datetime, timezone

def build_find_account_query(first_name, identifier, last_name=None):
    query = {
        "first_name": {"$regex": f"^{first_name}$", "$options": "i"},
        "$or": [
            {"phone_number": identifier},
            {"primary_email": identifier},
            {"recovery_phone": identifier},
            {"recovery_email": identifier},
        ]
    }
    if last_name:
        query["last_name"] = {"$regex": f"^{last_name}$", "$options": "i"}
    return query



@api_view(["POST"])
def find_account(request):
    first_name = request.data.get("first_name")
    last_name = request.data.get("last_name")
    identifier = request.data.get("identifier")

    if not first_name or not identifier:
        return Response(
            {"error": "first_name and identifier required"},
            status=400
        )

    try:
        validate_identifier(identifier)
        validate_name(first_name)
    except ValidationError as e:
         return Response(
            {"error": e.message},
            status=status.HTTP_400_BAD_REQUEST
        )
    
    if last_name:
        try:
            validate_name(last_name)
        except ValidationError as e:
            return Response(
            {"error": e.message},
            status=status.HTTP_400_BAD_REQUEST
        )
         
    query = build_find_account_query(first_name, identifier, last_name)
    users = list(users_collection.find(query))

    if not users:
        return Response({"error": "No accounts found"}, status=404)

    ip = request.META.get("REMOTE_ADDR")

    # ✅ Rate limit (ACCOUNT RECOVERY ONLY)
    allowed, error = can_send_account_otp(
        ip=ip,
        identifier=identifier,
        purpose="ACCOUNT_RECOVERY"
    )

    if not allowed:
        return Response({"error": error}, status=429)

    # 🔐 Generate OTP ONCE
    otp = generate_otp()
    otp_hash = hash_otp(otp)

    now = datetime.now(timezone.utc)

    recovery_token = str(uuid.uuid4())

    # ✅ SINGLE INSERT
    account_recovery_otp_collection.insert_one({
        "recovery_token": recovery_token,
        "identifier": identifier,
        "otp_hash": otp_hash,
        "matched_user_ids": [str(u["_id"]) for u in users],
        "attempts": 1,
        "ip_address": ip,
        "created_at": now,
        "expires_at": now + timedelta(minutes=10),
        "verified": False
    })

    # 📤 Send OTP
    if "@" in identifier:
        send_otp_email(
            to_email=identifier,
            otp=otp,
            sender_email=DEFAULT_FROM_EMAIL,
            sender_name="INDZS Security",
            purpose="Account Recovery"
        )
    else:
        send_otp_sms(identifier, otp, purpose="Account Recovery")

    return Response(
        {
            "message": "OTP sent",
            "recovery_token": recovery_token
        },
        status=200
    )


@api_view(["POST"])
def verify_find_account_otp(request):
    recovery_token = request.data.get("recovery_token")
    otp = request.data.get("otp")
    # print("Recovery Token")
    # print(recovery_token)
    # print(otp)
    
    try:
        validate_otp(otp)
    except ValidationError as e:
        return Response(
            {"error": e.message},
            status=status.HTTP_400_BAD_REQUEST
        )
        

    if not recovery_token or not otp:
        return Response(
            {"error": "recovery_token and otp required"},
            status=400
        )

    record = account_recovery_otp_collection.find_one({
        "recovery_token": recovery_token,
        "verified": False
    })

    if not record:
        return Response({"error": "Invalid or used OTP"}, status=400)

    now = datetime.now(timezone.utc)

    expires_at = record["expires_at"]
    if expires_at.tzinfo is None:
        expires_at = expires_at.replace(tzinfo=timezone.utc)

    if expires_at < now:
        return Response({"error": "OTP expired"}, status=400)

    if not verify_otp_hash(otp, record["otp_hash"]):
        return Response({"error": "Invalid OTP"}, status=400)

    # ✅ Mark verified (don’t delete immediately – safer)
    account_recovery_otp_collection.update_one(
        {"_id": record["_id"]},
        {"$set": {"verified": True}}
    )

    users = list(users_collection.find(
        {"_id": {"$in": [ObjectId(i) for i in record["matched_user_ids"]]}},
        {
            "password": 0,
            "recovery_email": 0,
            "recovery_phone": 0
        }
    ))

    return Response(
        {
            "accounts": [
                {
                    "user_id": str(u["_id"]),
                    "name": f"{u.get('first_name')} {u.get('last_name', '')}".strip(),
                    "masked_email": u.get("primary_email"),
                    "masked_phone": u.get("phone_number")
                }
                for u in users
            ]
        },
        status=200
    )




from rest_framework.decorators import api_view, parser_classes
from rest_framework.parsers import MultiPartParser, FormParser
from rest_framework.response import Response
from django.core.files.storage import default_storage
from .utils.process_profile_image import process_profile_image
import os

@api_view(["POST"])
@jwt_required
@parser_classes([MultiPartParser, FormParser])
def upload_profile_picture(request):
    file = request.FILES.get("profile_picture")

    if not file:
        return Response({"error": "profile_picture is required"}, status=400)

    try:
        processed_image = process_profile_image(file)
    except ValueError as e:
        return Response({"error": str(e)}, status=400)

    user_id = str(request.user_id)
    media_path = os.getenv("MEDIA_PATH")
    path = f"profile_pictures/user_{user_id}.jpg"

    if default_storage.exists(path):
        default_storage.delete(path)

    default_storage.save(path, processed_image)

    media_cdn_path = f"{media_path}/profile_pictures/user_{user_id}.jpg"

    users_collection.update_one(
        {"_id": ObjectId(user_id)},
        {"$set": {"profile_picture": media_cdn_path}}
    )

    return Response(
        {
            "message": "Profile picture updated",
            "profile_picture": media_cdn_path
        },
        status=200
    )





@api_view(["DELETE"])
@jwt_required
def remove_profile_picture(request):
    user_id = str(request.user_id)

    user = users_collection.find_one(
        {"_id": ObjectId(user_id)},
        {"profile_picture": 1}
    )

    if user and user.get("profile_picture"):
        path = user["profile_picture"]

        # 🧹 Delete file if exists
        if default_storage.exists(path):
            default_storage.delete(path)

        # 🧾 Remove reference from DB
        users_collection.update_one(
            {"_id": ObjectId(user_id)},
            {"$unset": {"profile_picture": ""}}
        )

    return Response(
        {"message": "Profile picture removed"},
        status=200
    )






@api_view(["POST"])
@jwt_required
def get_logged_in_devices(request):
    user_id = request.user_id

    # 🔐 Fetch all sessions for this user
    sessions = list(
        user_sessions_collection.find(
            {"user_id": ObjectId(user_id),
             "is_active": True,
             },
            {
                "_id": 0,  # hide mongo _id
                "session_id": 1,
                "device_id": 1,
                "browser": 1,
                "os": 1,
                "ip_address": 1,
                "created_at": 1,
                "last_active": 1,
                "is_active": 1,
                "logged_out_at": 1,
            }
        ).sort("created_at", -1)
    )

    total_devices = len(sessions)
    active_devices = sum(1 for s in sessions if s.get("is_active"))

    return Response(
        {
            "total_devices": total_devices,
            "active_devices": active_devices,
            "devices": sessions,
        },
        status=status.HTTP_200_OK
    )
